<?php
// Habilitar exibição de erros apenas em log
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'pix_errors.log');

// Incluir a classe PIX
require_once 'PIX.php';

header('Content-Type: application/json');

try {
    // Receber dados do POST
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Dados inválidos. Input recebido: ' . $input);
    }
    
    // Extrair dados
    $pixKey = $data['pixKey'] ?? '';
    $amount = floatval($data['amount'] ?? 0);
    $description = $data['description'] ?? '';
    
    // Log dos dados recebidos
    error_log("Dados recebidos: " . print_r($data, true));
    
    // Validar dados
    if (empty($pixKey)) {
        throw new Exception('Chave PIX é obrigatória');
    }
    
    if ($amount <= 0) {
        throw new Exception('Valor deve ser maior que zero');
    }

    // Formatar valor para duas casas decimais
    $amount = number_format($amount, 2, '.', '');

    // Limitar descrição a 25 caracteres (limite do PIX)
    $description = mb_substr($description, 0, 25);

    // Formatar a chave PIX
    $pixKey = trim($pixKey);
    
    // Remover formatação de CPF/CNPJ se necessário
    $cleanKey = preg_replace('/[^0-9a-zA-Z@._+-]/', '', $pixKey);
    
    // Validar formato da chave PIX
    if (filter_var($pixKey, FILTER_VALIDATE_EMAIL)) {
        // É um email, não precisa fazer nada
        $pixKey = strtolower($pixKey);
    } elseif (preg_match('/^\d{11}$/', $cleanKey)) {
        // É um CPF (11 dígitos)
        $pixKey = $cleanKey;
    } elseif (preg_match('/^\d{14}$/', $cleanKey)) {
        // É um CNPJ (14 dígitos)
        $pixKey = $cleanKey;
    } elseif (preg_match('/^\+?\d{12,13}$/', $cleanKey)) {
        // É um telefone com ou sem código do país
        $pixKey = '+55' . substr(preg_replace('/^\+55/', '', $cleanKey), -11);
    } elseif (preg_match('/^[0-9a-f]{8}-?[0-9a-f]{4}-?4[0-9a-f]{3}-?[89ab][0-9a-f]{3}-?[0-9a-f]{12}$/i', $cleanKey)) {
        // É uma chave aleatória (UUID)
        $pixKey = preg_replace('/[^0-9a-f]/i', '', $cleanKey);
        $pixKey = substr($pixKey, 0, 8) . '-' . 
                 substr($pixKey, 8, 4) . '-' . 
                 substr($pixKey, 12, 4) . '-' . 
                 substr($pixKey, 16, 4) . '-' . 
                 substr($pixKey, 20);
    } else {
        error_log("Chave PIX inválida: $pixKey (limpa: $cleanKey)");
        throw new Exception('Formato de chave PIX inválido. A chave deve ser um CPF, CNPJ, email, telefone ou chave aleatória.');
    }
    
    // Log da chave formatada
    error_log("Chave PIX formatada: $pixKey");
    
    // Gerar ID único para a transação
    $transactionId = uniqid('PIX_');
    
    // Configurar dados do merchant
    $merchantName = mb_strtoupper(preg_replace('/[^A-Za-z0-9 ]/', '', "MATEUS SHOP")); // Nome do merchant em maiúsculas, sem caracteres especiais
    $merchantCity = mb_strtoupper(preg_replace('/[^A-Za-z0-9 ]/', '', "SAO PAULO")); // Cidade em maiúsculas, sem caracteres especiais
    $merchantCep = preg_replace('/[^0-9]/', '', "25850000"); // CEP sem pontuação
    
    // Limitar tamanhos conforme especificação
    $merchantName = mb_substr($merchantName, 0, 25);
    $merchantCity = mb_substr($merchantCity, 0, 15);
    $merchantCep = substr($merchantCep, 0, 8);
    
    // Criar instância do PIX
    $pix = new PIX(
        $pixKey,
        $merchantName,
        $merchantCity,
        $merchantCep,
        $transactionId,
        $amount
    );
    
    // Gerar o payload do PIX
    $pixCode = $pix->payload();
    
    // Log do código PIX gerado
    error_log("Código PIX gerado: " . $pixCode);
    
    // Validar se o código PIX foi gerado corretamente
    if (empty($pixCode) || strlen($pixCode) < 10) {
        throw new Exception('Código PIX gerado é inválido');
    }

    // Validar formato EMV do código PIX
    if (!preg_match('/^00020101/', $pixCode)) {
        error_log("Código PIX inválido (formato EMV): " . $pixCode);
        throw new Exception('Código PIX gerado não está no formato EMV válido');
    }

    // Validar CRC16 no final do código
    if (!preg_match('/6304[A-F0-9]{4}$/', $pixCode)) {
        error_log("Código PIX inválido (CRC16): " . $pixCode);
        throw new Exception('Código PIX gerado não possui CRC16 válido');
    }
    
    // Retornar sucesso
    echo json_encode([
        'success' => true,
        'pixCode' => $pixCode
    ]);
    
} catch (Throwable $e) {
    // Log do erro
    error_log("Erro ao gerar PIX: " . $e->getMessage() . "\n" . $e->getTraceAsString());
    
    // Retornar erro
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao gerar código PIX: ' . $e->getMessage()
    ]);
} 