<?php
// Configuração do banco de dados SQLite
$db = new SQLite3('pix_pages.db');

// Verificar se as colunas existem e criar se necessário
$columns = [
    'logo_url' => "TEXT DEFAULT 'https://i.postimg.cc/85ydcNvK/MINHA-NOVA-LOGO-IPTV-THORTON.png'",
    'pix_logo_url' => "TEXT DEFAULT 'https://i.postimg.cc/m2q4v5d7/image.png'",
    'bank_info' => 'TEXT DEFAULT NULL',
    'pix_text' => "TEXT DEFAULT 'Sua Chave PIX:'"
];

// Obter lista de colunas existentes
$result = $db->query("PRAGMA table_info(pix_pages)");
$existing_columns = [];
while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
    $existing_columns[] = $row['name'];
}

// Adicionar colunas que não existem
foreach ($columns as $column => $definition) {
    if (!in_array($column, $existing_columns)) {
        $sql = "ALTER TABLE pix_pages ADD COLUMN $column $definition";
        $db->exec($sql);
    }
}

// Criar tabela se não existir
$db->exec('CREATE TABLE IF NOT EXISTS pix_pages (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    filename TEXT NOT NULL,
    title TEXT NOT NULL,
    pix_key TEXT NOT NULL,
    pix_type TEXT NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
)');

// Criar tabela de planos se não existir
$db->exec('CREATE TABLE IF NOT EXISTS planos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    nome TEXT NOT NULL,
    valor REAL NOT NULL,
    descricao TEXT NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
)');

// Função para salvar/atualizar página
function savePage($filename, $title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText) {
    global $db;
    
    // Primeiro, excluir o registro existente
    $db->exec("DELETE FROM pix_pages WHERE filename = '" . SQLite3::escapeString($filename) . "'");
    
    // Depois, inserir o novo registro
    $stmt = $db->prepare('INSERT INTO pix_pages (filename, title, pix_key, pix_type, logo_url, pix_logo_url, bank_info, pix_text) 
                         VALUES (:filename, :title, :pix_key, :pix_type, :logo_url, :pix_logo_url, :bank_info, :pix_text)');
    $stmt->bindValue(':filename', $filename, SQLITE3_TEXT);
    $stmt->bindValue(':title', $title, SQLITE3_TEXT);
    $stmt->bindValue(':pix_key', $pixKey, SQLITE3_TEXT);
    $stmt->bindValue(':pix_type', $pixType, SQLITE3_TEXT);
    $stmt->bindValue(':logo_url', $logoUrl, SQLITE3_TEXT);
    $stmt->bindValue(':pix_logo_url', $pixLogoUrl, SQLITE3_TEXT);
    $stmt->bindValue(':bank_info', $bankInfo, SQLITE3_TEXT);
    $stmt->bindValue(':pix_text', $pixText, SQLITE3_TEXT);
    return $stmt->execute();
}

// Função para carregar planos do banco de dados
function loadPlanos() {
    global $db;
    $planos = [];
    $result = $db->query('SELECT * FROM planos ORDER BY id ASC');
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $planos[] = [
            'nome' => $row['nome'],
            'valor' => number_format($row['valor'], 2, '.', ''),
            'descricao' => $row['descricao']
        ];
    }
    return $planos;
}

// Função para salvar planos no banco de dados
function savePlanos($planos) {
    global $db;
    
    // Começar uma transação
    $db->exec('BEGIN TRANSACTION');
    
    try {
        // Limpar tabela existente
        $db->exec('DELETE FROM planos');
        
        // Inserir novos planos
        $stmt = $db->prepare('INSERT INTO planos (nome, valor, descricao) VALUES (:nome, :valor, :descricao)');
        foreach ($planos as $plano) {
            $stmt->bindValue(':nome', $plano['nome'], SQLITE3_TEXT);
            $stmt->bindValue(':valor', floatval($plano['valor']), SQLITE3_FLOAT);
            $stmt->bindValue(':descricao', $plano['descricao'], SQLITE3_TEXT);
            $stmt->execute();
        }
        
        // Confirmar transação
        $db->exec('COMMIT');
        return true;
    } catch (Exception $e) {
        // Reverter em caso de erro
        $db->exec('ROLLBACK');
        return false;
    }
}

// Função para gerar o HTML da página PIX
function generatePixPage($title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText) {
    global $db;
    
    // Carregar planos do banco de dados
    $planos = [];
    $result = $db->query('SELECT * FROM planos ORDER BY id ASC');
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $planos[] = [
            'nome' => $row['nome'],
            'valor' => number_format($row['valor'], 2, '.', ''),
            'descricao' => $row['descricao']
        ];
    }
    
    // Se não houver planos, criar um plano padrão
    if (empty($planos)) {
        $planos = [[
            'nome' => 'Plano Básico',
            'valor' => '49.90',
            'descricao' => 'Acesso básico'
        ]];
        
        // Salvar plano padrão no banco
        $stmt = $db->prepare('INSERT INTO planos (nome, valor, descricao) VALUES (:nome, :valor, :descricao)');
        $stmt->bindValue(':nome', 'Plano Básico', SQLITE3_TEXT);
        $stmt->bindValue(':valor', 49.90, SQLITE3_FLOAT);
        $stmt->bindValue(':descricao', 'Acesso básico', SQLITE3_TEXT);
        $stmt->execute();
    }
    
    // Garantir que os planos sejam codificados corretamente para JavaScript
    $planosJson = json_encode($planos, JSON_HEX_QUOT | JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_UNESCAPED_UNICODE);
    if ($planosJson === false) {
        error_log("Erro ao codificar planos para JSON: " . json_last_error_msg());
        $planosJson = '[]';
    }
    
    // Remover caracteres especiais que podem causar problemas no JavaScript
    $planosJson = str_replace(["\r", "\n", "\t"], '', $planosJson);
    
    $template = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>' . htmlspecialchars($title) . '</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body { 
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        .pix-container {
            max-width: 600px;
            background: white;
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            padding: 2rem;
            margin: 2rem auto;
        }
        .logo-container {
            background: white;
            padding: 1rem;
            border-radius: 10px;
            margin-bottom: 2rem;
        }
        .pix-key-container {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 1rem;
            margin: 1rem 0;
        }
        #pixKey {
            font-size: 0.9rem !important;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 100%;
        }
        .copy-button, .plan-button, .qr-button {
            background: #0d6efd;
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 10px;
            font-weight: bold;
            transition: all 0.3s ease;
            margin: 0.5rem;
            display: inline-block;
        }
        .copy-button:hover, .plan-button:hover, .qr-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            color: white;
            text-decoration: none;
        }
        .plan-button {
            background: #198754;
        }
        .qr-button {
            background: #6c757d;
        }
        .pix-type-badge {
            background: #198754;
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.9rem;
            margin-bottom: 1rem;
            display: inline-block;
        }
        .bank-info {
            background: #e9ecef;
            padding: 1rem;
            border-radius: 10px;
            margin: 1rem 0;
            font-size: 0.9rem;
            color: #495057;
        }
        .plan-card {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 1rem;
            margin: 0.5rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .plan-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            border-color: #0d6efd;
        }
        .plan-card.selected {
            border-color: #0d6efd;
            background: #f8f9fa;
        }
        #copyButton, #qrButton {
            display: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="pix-container">
            <div class="logo-container text-center">
                <img src="' . htmlspecialchars($logoUrl) . '" alt="Logo" class="img-fluid" style="max-height: 100px;">
            </div>
            
            <div class="text-center">
                <div class="pix-type-badge">
                    <i class="fas fa-qrcode me-2"></i>PIX COPIA E COLA
                </div>
                
                <div class="pix-key-container">
                    <h5 class="mb-3">PIX COPIA E COLA</h5>
                    <input type="text" id="pixKey" class="form-control form-control-lg text-center mb-3" 
                           value="ESCOLHA UM DE NOSSOS PLANOS PARA GERAR O PIX" 
                           data-pix-key="' . htmlspecialchars($pixKey) . '" readonly>
                    
                    <button class="plan-button" onclick="showPlansModal()">
                        <i class="fas fa-list me-2"></i>PLANOS
                    </button>

                    <button id="copyButton" class="copy-button" onclick="copiarChavePIX()">
                        <i class="fas fa-copy me-2"></i>COPIAR PIX COPIA E COLA
                    </button>

                    <button id="qrButton" class="qr-button" onclick="showQRCode()">
                        <i class="fas fa-qrcode me-2"></i>QR CODE
                    </button>
                </div>
                
                ' . ($bankInfo ? '<div class="bank-info text-center">
                    <i class="fas fa-university me-2"></i>' . nl2br(htmlspecialchars($bankInfo)) . '
                </div>' : '') . '
                
                <p class="text-danger mt-4">
                    <i class="fas fa-info-circle me-2"></i>
                    Após realizar o PIX, envie o comprovante para nosso WhatsApp
                </p>
                
                <img src="' . htmlspecialchars($pixLogoUrl) . '" alt="Logo PIX" class="img-fluid mt-4" style="max-width: 200px;">
            </div>
        </div>
    </div>

    <!-- Modal de Planos -->
    <div class="modal fade" id="plansModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Escolha seu Plano</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row" id="plansContainer">
                        <!-- Planos serão inseridos aqui via JavaScript -->
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal do QR Code -->
    <div class="modal fade" id="qrModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">QR Code PIX</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body text-center">
                    <img id="qrCodeImage" src="" alt="QR Code" class="img-fluid" style="max-width: 500px;">
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery.mask/1.14.16/jquery.mask.min.js"></script>
    <?php if (isset($message)): ?>
    <script>var systemMessage = <?php echo json_encode($message); ?>;</script>
    <?php endif; ?>
    <script>
    // Definir planos no escopo global
    window.planos = ' . $planosJson . ';
    console.log("Planos carregados:", window.planos);
    </script>
    <script src="script.js"></script>
</body>
</html>';
    
    return $template;
}

// Processar ações do formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $filename = $_POST['filename'];
                if (substr($filename, -5) !== '.html') {
                    $filename .= '.html';
                }
                $title = $_POST['title'];
                $pixKey = $_POST['pix_key'];
                $pixType = $_POST['pix_type'];
                $logoUrl = $_POST['logo_url'];
                $pixLogoUrl = $_POST['pix_logo_url'];
                $bankInfo = $_POST['bank_info'];
                $pixText = $_POST['pix_text'];
                
                // Salvar no banco de dados
                savePage($filename, $title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText);
                
                // Gerar e salvar arquivo HTML
                $html = generatePixPage($title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText);
                file_put_contents($filename, $html);
                $message = "Página criada com sucesso!";
                break;
                
            case 'edit':
                $filename = $_POST['filename'];
                $title = $_POST['title'];
                $pixKey = $_POST['pix_key'];
                $pixType = $_POST['pix_type'];
                $logoUrl = $_POST['logo_url'];
                $pixLogoUrl = $_POST['pix_logo_url'];
                $bankInfo = $_POST['bank_info'];
                $pixText = $_POST['pix_text'];
                
                // Atualizar no banco de dados
                savePage($filename, $title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText);
                
                // Atualizar arquivo HTML
                $html = generatePixPage($title, $pixKey, $pixType, $logoUrl, $pixLogoUrl, $bankInfo, $pixText);
                file_put_contents($filename, $html);
                $message = "Página atualizada com sucesso!";
                break;

            case 'update_logos':
                $logoUrl = $_POST['global_logo_url'];
                $pixLogoUrl = $_POST['global_pix_logo_url'];
                $success = 0;
                $failed = 0;
                
                // Atualizar todas as páginas
                $result = $db->query('SELECT * FROM pix_pages');
                while ($page = $result->fetchArray(SQLITE3_ASSOC)) {
                    try {
                        // Atualizar no banco
                        $stmt = $db->prepare('UPDATE pix_pages SET logo_url = :logo_url, pix_logo_url = :pix_logo_url WHERE filename = :filename');
                        $stmt->bindValue(':logo_url', $logoUrl, SQLITE3_TEXT);
                        $stmt->bindValue(':pix_logo_url', $pixLogoUrl, SQLITE3_TEXT);
                        $stmt->bindValue(':filename', $page['filename'], SQLITE3_TEXT);
                        $stmt->execute();
                        
                        // Atualizar arquivo HTML
                        $html = generatePixPage($page['title'], $page['pix_key'], $page['pix_type'], $logoUrl, $pixLogoUrl, $page['bank_info'], $page['pix_text']);
                        if (file_put_contents($page['filename'], $html) !== false) {
                            $success++;
                        } else {
                            $failed++;
                        }
                    } catch (Exception $e) {
                        $failed++;
                    }
                }
                
                $message = "Logos atualizadas! $success páginas atualizadas com sucesso.";
                if ($failed > 0) {
                    $message .= " $failed páginas não puderam ser atualizadas.";
                }
                break;

            case 'update_pix_text':
                $newPixText = $_POST['global_pix_text'];
                $success = 0;
                $failed = 0;
                
                // Atualizar todas as páginas
                $result = $db->query('SELECT * FROM pix_pages');
                while ($page = $result->fetchArray(SQLITE3_ASSOC)) {
                    try {
                        // Atualizar no banco
                        $stmt = $db->prepare('UPDATE pix_pages SET pix_text = :pix_text WHERE filename = :filename');
                        $stmt->bindValue(':pix_text', $newPixText, SQLITE3_TEXT);
                        $stmt->bindValue(':filename', $page['filename'], SQLITE3_TEXT);
                        $stmt->execute();
                        
                        // Atualizar arquivo HTML
                        $html = generatePixPage($page['title'], $page['pix_key'], $page['pix_type'], 
                                             $page['logo_url'], $page['pix_logo_url'], 
                                             $page['bank_info'], $newPixText);
                        if (file_put_contents($page['filename'], $html) !== false) {
                            $success++;
                        } else {
                            $failed++;
                        }
                    } catch (Exception $e) {
                        $failed++;
                    }
                }
                
                $message = "Texto PIX atualizado! $success páginas atualizadas com sucesso.";
                if ($failed > 0) {
                    $message .= " $failed páginas não puderam ser atualizadas.";
                }
                break;

            case 'update_all_pages':
                $success = 0;
                $failed = 0;
                
                // Buscar todas as páginas do banco
                $result = $db->query('SELECT * FROM pix_pages');
                while ($page = $result->fetchArray(SQLITE3_ASSOC)) {
                    try {
                        // Verificar se o arquivo existe
                        if (!file_exists($page['filename'])) {
                            $failed++;
                            continue;
                        }

                        // Fazer backup do arquivo original
                        $backupFile = $page['filename'] . '.bak';
                        if (file_exists($page['filename'])) {
                            copy($page['filename'], $backupFile);
                        }

                        // Gerar novo HTML com as novas funcionalidades
                        $html = generatePixPage(
                            $page['title'],
                            $page['pix_key'],
                            'PIX COPIA E COLA', // Forçar o tipo para PIX COPIA E COLA
                            $page['logo_url'] ?: "https://i.postimg.cc/85ydcNvK/MINHA-NOVA-LOGO-IPTV-THORTON.png",
                            $page['pix_logo_url'] ?: "https://i.postimg.cc/m2q4v5d7/image.png",
                            $page['bank_info'] ? $page['bank_info'] : "",
                            "PIX COPIA E COLA"
                        );
                        
                        // Salvar o novo arquivo
                        if (file_put_contents($page['filename'], $html) !== false) {
                            // Atualizar o banco de dados com os novos valores
                            $stmt = $db->prepare('UPDATE pix_pages SET pix_type = :pix_type, pix_text = :pix_text WHERE filename = :filename');
                            $stmt->bindValue(':pix_type', 'PIX COPIA E COLA', SQLITE3_TEXT);
                            $stmt->bindValue(':pix_text', 'PIX COPIA E COLA', SQLITE3_TEXT);
                            $stmt->bindValue(':filename', $page['filename'], SQLITE3_TEXT);
                            $stmt->execute();
                            
                            $success++;
                            // Remover backup se a atualização foi bem sucedida
                            if (file_exists($backupFile)) {
                                unlink($backupFile);
                            }
                        } else {
                            $failed++;
                            // Restaurar backup se a atualização falhou
                            if (file_exists($backupFile)) {
                                copy($backupFile, $page['filename']);
                                unlink($backupFile);
                            }
                        }
                    } catch (Exception $e) {
                        $failed++;
                        // Restaurar backup se ocorreu um erro
                        if (file_exists($backupFile)) {
                            copy($backupFile, $page['filename']);
                            unlink($backupFile);
                        }
                    }
                }
                
                $message = "Atualização concluída! $success páginas atualizadas com sucesso.";
                if ($failed > 0) {
                    $message .= " $failed páginas não puderam ser atualizadas.";
                }
                break;

            case 'delete':
                $filename = $_POST['filename'];
                if (file_exists($filename)) {
                    unlink($filename);
                    $db->exec("DELETE FROM pix_pages WHERE filename = '" . SQLite3::escapeString($filename) . "'");
                    $message = "Página excluída com sucesso!";
                }
                break;

            case 'update_all_styles':
                $success = 0;
                $failed = 0;
                
                // Buscar todas as páginas do banco
                $result = $db->query('SELECT * FROM pix_pages');
                while ($page = $result->fetchArray(SQLITE3_ASSOC)) {
                    try {
                        // Gerar novo HTML com o estilo atual
                        $html = generatePixPage($page['title'], $page['pix_key'], $page['pix_type'], $page['logo_url'], $page['pix_logo_url'], $page['bank_info'], $page['pix_text']);
                        
                        // Salvar o arquivo
                        if (file_put_contents($page['filename'], $html) !== false) {
                            $success++;
                        } else {
                            $failed++;
                        }
                    } catch (Exception $e) {
                        $failed++;
                    }
                }
                
                $message = "Atualização de estilos concluída! $success páginas atualizadas com sucesso.";
                if ($failed > 0) {
                    $message .= " $failed páginas não puderam ser atualizadas.";
                }
                break;

            case 'delete_selected':
                $filenames = isset($_POST['selected_pages']) ? $_POST['selected_pages'] : [];
                $deleted = 0;
                
                foreach ($filenames as $filename) {
                    if (file_exists($filename)) {
                        unlink($filename);
                        $db->exec("DELETE FROM pix_pages WHERE filename = '" . SQLite3::escapeString($filename) . "'");
                        $deleted++;
                    }
                }
                
                $message = "$deleted páginas foram excluídas com sucesso!";
                break;

            case 'save_planos':
                $planos = [];
                $nomes = $_POST['plano_nome'] ?? [];
                $valores = $_POST['plano_valor'] ?? [];
                $descricoes = $_POST['plano_descricao'] ?? [];
                
                foreach ($nomes as $i => $nome) {
                    if (!empty($nome) && isset($valores[$i]) && isset($descricoes[$i])) {
                        // Converter o valor para float
                        $valor = floatval(str_replace(',', '.', str_replace('.', '', $valores[$i]))) / 100;
                        $planos[] = [
                            'nome' => $nome,
                            'valor' => number_format($valor, 2, '.', ''),
                            'descricao' => $descricoes[$i]
                        ];
                    }
                }
                
                if (savePlanos($planos)) {
                    $message = 'Planos salvos com sucesso!';
                    $messageType = 'success';
                } else {
                    $message = 'Erro ao salvar os planos. Por favor, tente novamente.';
                    $messageType = 'error';
                }
                break;
        }
    }
}

// Buscar páginas existentes
$pages = [];
$result = $db->query('SELECT DISTINCT filename, title, pix_key, pix_type, logo_url, pix_logo_url, bank_info, created_at 
                      FROM pix_pages 
                      GROUP BY filename 
                      ORDER BY created_at DESC');
while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
    $pages[] = $row;
}

// Importar páginas HTML existentes que ainda não estão no banco
$htmlFiles = glob("*.html");
foreach ($htmlFiles as $file) {
    // Verifica se o arquivo já existe no banco de dados
    $stmt = $db->prepare('SELECT COUNT(*) as count FROM pix_pages WHERE filename = :filename');
    $stmt->bindValue(':filename', $file, SQLITE3_TEXT);
    $result = $stmt->execute();
    $count = $result->fetchArray(SQLITE3_ASSOC)['count'];
    
    if ($count == 0) {
        // Tenta extrair informações do arquivo HTML
        $content = file_get_contents($file);
        
        // Extrair título
        preg_match('/<title[^>]*>(.*?)<\/title>/i', $content, $titleMatches);
        $title = isset($titleMatches[1]) ? strip_tags($titleMatches[1]) : "PIX Page";
        
        // Extrair chave PIX
        preg_match('/value="([^"]*)"[^>]*id="email"/i', $content, $keyMatches);
        $pixKey = isset($keyMatches[1]) ? $keyMatches[1] : "Chave não definida";
        
        // Extrair tipo de PIX do texto de descrição
        preg_match('/PIX\s+(E-MAIL|EMAIL|CPF|CNPJ|CELULAR|ALEATORIA)/i', $content, $typeMatches);
        $pixType = isset($typeMatches[1]) ? str_replace('E-MAIL', 'EMAIL', strtoupper($typeMatches[1])) : "EMAIL";
        
        // Adicionar ao banco de dados
        savePage($file, $title, $pixKey, $pixType, "https://i.postimg.cc/85ydcNvK/MINHA-NOVA-LOGO-IPTV-THORTON.png", "https://i.postimg.cc/m2q4v5d7/image.png", "", "Sua Chave PIX:");
    }
}

// Recarregar páginas após importação
$pages = [];
$result = $db->query('SELECT DISTINCT filename, title, pix_key, pix_type, logo_url, pix_logo_url, bank_info, created_at 
                      FROM pix_pages 
                      GROUP BY filename 
                      ORDER BY created_at DESC');
while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
    $pages[] = $row;
}
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Painel de Gerenciamento PIX</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link href="styles.css" rel="stylesheet">
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary mb-4">
        <div class="container">
            <a class="navbar-brand" href="#">
                <i class="fas fa-qrcode me-2"></i>
                Gerenciador PIX
            </a>
            <button class="btn btn-outline-light ms-auto" onclick="location.reload()">
                <i class="fas fa-sync-alt me-2"></i>Atualizar
            </button>
        </div>
    </nav>

    <!-- Modal de Edição -->
    <div class="modal fade" id="editModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Página PIX</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editForm" method="POST">
                        <input type="hidden" name="action" value="edit">
                        <input type="hidden" name="filename" id="edit_filename">
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Título da Página</label>
                                <input type="text" name="title" id="edit_title" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Tipo de Chave</label>
                                <select name="pix_type" id="edit_pix_type" class="form-select" required>
                                    <option value="EMAIL">E-mail</option>
                                    <option value="CPF">CPF</option>
                                    <option value="CNPJ">CNPJ</option>
                                    <option value="CELULAR">Celular</option>
                                    <option value="ALEATORIA">Chave Aleatória</option>
                                </select>
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Chave PIX</label>
                                <input type="text" name="pix_key" id="edit_pix_key" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">URL da Logo Principal</label>
                                <input type="url" name="logo_url" id="edit_logo_url" class="form-control" 
                                       onchange="previewImage(this, 'edit_logo_preview')" required>
                                <img id="edit_logo_preview" class="preview-image">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">URL da Logo PIX</label>
                                <input type="url" name="pix_logo_url" id="edit_pix_logo_url" class="form-control" 
                                       onchange="previewImage(this, 'edit_pix_logo_preview')" required>
                                <img id="edit_pix_logo_preview" class="preview-image">
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Informações do Banco (opcional)</label>
                                <textarea name="bank_info" id="edit_bank_info" class="form-control" 
                                          rows="3" placeholder="Ex: Banco XYZ - Ag: 0000 - Conta: 00000-0"></textarea>
                            </div>

                            <div class="col-12">
                                <label class="form-label">Texto da Chave PIX</label>
                                <input type="text" name="pix_text" id="edit_pix_text" class="form-control" required>
                            </div>
                        </div>
                        
                        <div class="text-end mt-3">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                            <button type="submit" class="btn btn-primary">Salvar Alterações</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="container py-4">
        <!-- Contador de Seleção -->
        <div id="selected-counter" class="alert alert-info text-center mb-4" style="display: none;">
            0 página(s) selecionada(s)
        </div>

        <div class="row">
            <div class="col-12">
                <!-- Cards existentes com tooltips -->
                <div class="card mb-4" data-tooltip="Atualize as logos em todas as páginas de uma vez">
                    <div class="card-header bg-success text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-images me-2"></i>Atualizar Logos em Todas as Páginas
                        </h4>
                    </div>
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <input type="hidden" name="action" value="update_logos">
                            
                            <div class="col-md-6">
                                <label class="form-label">Nova URL da Logo Principal</label>
                                <input type="url" name="global_logo_url" class="form-control" 
                                       onchange="previewImage(this, 'global_logo_preview')" required>
                                <img id="global_logo_preview" class="preview-image">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Nova URL da Logo PIX</label>
                                <input type="url" name="global_pix_logo_url" class="form-control" 
                                       onchange="previewImage(this, 'global_pix_logo_preview')" required>
                                <img id="global_pix_logo_preview" class="preview-image">
                            </div>
                            
                            <div class="col-12">
                                <button type="submit" class="btn btn-success" onclick="return confirm('Isso irá atualizar as logos em todas as páginas. Deseja continuar?')">
                                    <i class="fas fa-sync-alt me-2"></i>Atualizar Logos em Todas as Páginas
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Card de Atualização Global do Texto PIX -->
                <div class="card mb-4" data-tooltip="Atualize o texto da chave PIX em todas as páginas">
                    <div class="card-header bg-success text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-font me-2"></i>Atualizar Texto PIX em Todas as Páginas
                        </h4>
                    </div>
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <input type="hidden" name="action" value="update_pix_text">
                            
                            <div class="col-12">
                                <label class="form-label">Novo Texto da Chave PIX</label>
                                <input type="text" name="global_pix_text" class="form-control" 
                                       value="Sua Chave PIX:" required>
                            </div>
                            
                            <div class="col-12">
                                <button type="submit" class="btn btn-success" onclick="return confirm('Isso irá atualizar o texto PIX em todas as páginas. Deseja continuar?')">
                                    <i class="fas fa-sync-alt me-2"></i>Atualizar Texto PIX em Todas as Páginas
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Card de Atualização de Todas as Páginas -->
                <div class="card mb-4" data-tooltip="Atualiza todas as páginas com as novas funcionalidades">
                    <div class="card-header bg-warning text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-sync me-2"></i>Atualizar Todas as Páginas
                        </h4>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="action" value="update_all_pages">
                            
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                Esta ação irá atualizar todas as páginas PIX com as novas funcionalidades:
                                <ul class="mb-0 mt-2">
                                    <li>Botão de Planos</li>
                                    <li>PIX Copia e Cola</li>
                                    <li>QR Code</li>
                                    <li>Novo layout e melhorias visuais</li>
                                </ul>
                            </div>
                            
                            <button type="submit" class="btn btn-warning" onclick="return confirm('Isso irá atualizar todas as páginas com as novas funcionalidades. Deseja continuar?')">
                                <i class="fas fa-sync-alt me-2"></i>Atualizar Todas as Páginas
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Card de Gerenciamento de Planos -->
                <div class="card mt-4">
                    <div class="card-header bg-success text-white">
                        <h5 class="card-title mb-0">Gerenciar Planos</h5>
                    </div>
                    <div class="card-body">
                        <form id="planosForm" method="post" action="">
                            <input type="hidden" name="action" value="save_planos">
                            
                            <div id="planos-container">
                                <?php
                                $planos = loadPlanos();
                                if (empty($planos)) {
                                    // Plano padrão se não houver planos cadastrados
                                    $planos = [[
                                        'nome' => '',
                                        'valor' => '',
                                        'descricao' => ''
                                    ]];
                                }
                                
                                foreach ($planos as $plano) {
                                    ?>
                                    <div class="row mb-3 plano-item">
                                        <div class="col-md-4">
                                            <label class="form-label">Nome do Plano</label>
                                            <input type="text" name="plano_nome[]" class="form-control" 
                                                   value="<?php echo htmlspecialchars($plano['nome']); ?>" required>
                                        </div>
                                        <div class="col-md-3">
                                            <label class="form-label">Valor (R$)</label>
                                            <input type="text" name="plano_valor[]" class="form-control money" 
                                                   value="<?php echo $plano['valor'] ? number_format(floatval($plano['valor']), 2, ',', '.') : ''; ?>" required>
                                            <small class="text-muted">Digite o valor (ex: 35,00 para R$ 35,00)</small>
                                        </div>
                                        <div class="col-md-4">
                                            <label class="form-label">Descrição</label>
                                            <input type="text" name="plano_descricao[]" class="form-control" 
                                                   value="<?php echo htmlspecialchars($plano['descricao']); ?>" required>
                                        </div>
                                        <div class="col-md-1 d-flex align-items-end">
                                            <button type="button" class="btn btn-danger btn-sm" onclick="removePlano(this)">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                    <?php
                                }
                                ?>
                            </div>
                            
                            <div class="row mt-3">
                                <div class="col-12">
                                    <button type="button" class="btn btn-primary" id="addPlanoBtn">
                                        <i class="fas fa-plus me-2"></i>Adicionar Plano
                                    </button>
                                    <button type="submit" class="btn btn-success">
                                        <i class="fas fa-save me-2"></i>Salvar Planos
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Card de Criação com validação -->
                <div class="card mb-4" data-tooltip="Crie uma nova página PIX">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-plus-circle me-2"></i>Criar Nova Página PIX
                        </h4>
                    </div>
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <input type="hidden" name="action" value="create">
                            
                            <div class="col-md-6">
                                <label class="form-label">Nome do Arquivo</label>
                                <input type="text" name="filename" class="form-control" required 
                                       placeholder="exemplo.html">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Título da Página</label>
                                <input type="text" name="title" class="form-control" required 
                                       placeholder="Minha Página PIX">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Chave PIX</label>
                                <input type="text" name="pix_key" class="form-control" required 
                                       placeholder="Sua chave PIX">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Tipo de Chave</label>
                                <select name="pix_type" class="form-select" required>
                                    <option value="EMAIL">E-mail</option>
                                    <option value="CPF">CPF</option>
                                    <option value="CNPJ">CNPJ</option>
                                    <option value="CELULAR">Celular</option>
                                    <option value="ALEATORIA">Chave Aleatória</option>
                                </select>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">URL da Logo Principal</label>
                                <input type="url" name="logo_url" class="form-control" 
                                       value="https://i.postimg.cc/85ydcNvK/MINHA-NOVA-LOGO-IPTV-THORTON.png"
                                       onchange="previewImage(this, 'create_logo_preview')" required>
                                <img id="create_logo_preview" class="preview-image" 
                                     src="https://i.postimg.cc/85ydcNvK/MINHA-NOVA-LOGO-IPTV-THORTON.png">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">URL da Logo PIX</label>
                                <input type="url" name="pix_logo_url" class="form-control" 
                                       value="https://i.postimg.cc/m2q4v5d7/image.png"
                                       onchange="previewImage(this, 'create_pix_logo_preview')" required>
                                <img id="create_pix_logo_preview" class="preview-image" 
                                     src="https://i.postimg.cc/m2q4v5d7/image.png">
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Informações do Banco (opcional)</label>
                                <textarea name="bank_info" class="form-control" rows="3" 
                                          placeholder="Ex: Banco XYZ - Ag: 0000 - Conta: 00000-0"></textarea>
                            </div>

                            <div class="col-12">
                                <label class="form-label">Texto da Chave PIX</label>
                                <input type="text" name="pix_text" class="form-control" 
                                       value="Sua Chave PIX:" required>
                            </div>
                            
                            <div class="col-12">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i>Criar Página
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Lista de Páginas com melhorias visuais -->
                <div class="card" data-tooltip="Gerencie suas páginas PIX existentes">
                    <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="fas fa-list me-2"></i>Páginas Existentes
                        </h4>
                        <button class="btn btn-light btn-sm" onclick="location.reload()">
                            <i class="fas fa-sync-alt me-2"></i>Atualizar Lista
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <!-- Botão de exclusão em massa (inicialmente oculto) -->
                            <div id="bulk-actions" class="mb-3" style="display: none;">
                                <form id="delete-selected-form" method="POST">
                                    <input type="hidden" name="action" value="delete_selected">
                                    <button type="button" class="btn btn-danger" onclick="confirmDeleteSelected()">
                                        <i class="fas fa-trash me-2"></i>Excluir Selecionados
                                    </button>
                                </form>
                            </div>
                            
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>
                                            <input type="checkbox" id="select-all" onchange="toggleAllCheckboxes()">
                                        </th>
                                        <th>Arquivo</th>
                                        <th>Título</th>
                                        <th>Chave PIX</th>
                                        <th>Tipo</th>
                                        <th>Banco</th>
                                        <th>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($pages as $page): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="selected_pages[]" 
                                                   value="<?php echo htmlspecialchars($page['filename']); ?>"
                                                   class="page-checkbox" onchange="updateBulkActions()">
                                        </td>
                                        <td><?php echo htmlspecialchars($page['filename']); ?></td>
                                        <td><?php echo htmlspecialchars($page['title']); ?></td>
                                        <td><?php echo htmlspecialchars($page['pix_key']); ?></td>
                                        <td><span class="badge bg-success"><?php echo htmlspecialchars($page['pix_type']); ?></span></td>
                                        <td><?php echo $page['bank_info'] ? htmlspecialchars($page['bank_info']) : '<span class="text-muted">-</span>'; ?></td>
                                        <td class="action-buttons">
                                            <button class="btn btn-sm btn-warning" onclick="editPage('<?php 
                                                echo htmlspecialchars(addslashes($page['filename'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['title'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['pix_key'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['pix_type'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['logo_url'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['pix_logo_url'])); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['bank_info'] ?? '')); ?>', '<?php 
                                                echo htmlspecialchars(addslashes($page['pix_text'] ?? 'Sua Chave PIX:')); ?>')">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <a href="<?php echo htmlspecialchars($page['filename']); ?>" 
                                               class="btn btn-sm btn-info" target="_blank">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="filename" 
                                                       value="<?php echo htmlspecialchars($page['filename']); ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" 
                                                        onclick="return confirm('Tem certeza que deseja excluir esta página?')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery.mask/1.14.16/jquery.mask.min.js"></script>
    <?php if (isset($message)): ?>
    <script>var systemMessage = <?php echo json_encode($message); ?>;</script>
    <?php endif; ?>
    <script src="script.js"></script>
</body>
</html> 