// Importar a biblioteca PixJS
import { CopyAndPaste, QrCodeImage } from '/node_modules/pixjs/dist/index.js';

// Funções de Utilidade
const showLoading = () => {
    Swal.fire({
        title: 'Processando...',
        html: '<div class="loading-spinner mx-auto"></div>',
        showConfirmButton: false,
        allowOutsideClick: false
    });
};

const showMessage = (title, text, icon = 'success', timer = 3000) => {
    Swal.fire({
        title,
        text,
        icon,
        timer,
        timerProgressBar: true,
        showConfirmButton: false
    });
};

// Preview de Imagens
function previewImage(input, previewId) {
    const preview = document.getElementById(previewId);
    preview.src = input.value;
    preview.style.display = 'block';
    
    // Verificar se a imagem carrega corretamente
    preview.onerror = () => {
        preview.src = 'https://via.placeholder.com/150?text=Imagem+Inválida';
        showMessage('Aviso', 'URL da imagem inválida ou inacessível', 'warning');
    };
}

// Funções de Edição
function editPage(filename, title, pixKey, pixType, logoUrl, pixLogoUrl, bankInfo, pixText) {
    document.getElementById('edit_filename').value = filename;
    document.getElementById('edit_title').value = title;
    document.getElementById('edit_pix_key').value = pixKey;
    document.getElementById('edit_pix_type').value = pixType;
    document.getElementById('edit_logo_url').value = logoUrl;
    document.getElementById('edit_pix_logo_url').value = pixLogoUrl;
    document.getElementById('edit_bank_info').value = bankInfo || '';
    document.getElementById('edit_pix_text').value = pixText || 'Sua Chave PIX:';
    
    // Atualizar previews
    previewImage(document.getElementById('edit_logo_url'), 'edit_logo_preview');
    previewImage(document.getElementById('edit_pix_logo_url'), 'edit_pix_logo_preview');
    
    new bootstrap.Modal(document.getElementById('editModal')).show();
}

// Gerenciamento de Checkboxes
function updateBulkActions() {
    const checkboxes = document.querySelectorAll('.page-checkbox:checked');
    const bulkActions = document.getElementById('bulk-actions');
    const selectAll = document.getElementById('select-all');
    const totalCheckboxes = document.querySelectorAll('.page-checkbox').length;
    
    // Atualizar visibilidade das ações em massa
    bulkActions.style.display = checkboxes.length > 0 ? 'block' : 'none';
    
    // Atualizar estado do checkbox principal
    selectAll.checked = checkboxes.length === totalCheckboxes;
    selectAll.indeterminate = checkboxes.length > 0 && checkboxes.length < totalCheckboxes;
    
    // Atualizar contador de seleção
    const counter = document.getElementById('selected-counter');
    if (counter) {
        counter.textContent = `${checkboxes.length} página(s) selecionada(s)`;
    }
}

function toggleAllCheckboxes() {
    const selectAll = document.getElementById('select-all');
    const checkboxes = document.querySelectorAll('.page-checkbox');
    checkboxes.forEach(checkbox => checkbox.checked = selectAll.checked);
    updateBulkActions();
}

// Confirmações
function confirmDeleteSelected() {
    const checkboxes = document.querySelectorAll('.page-checkbox:checked');
    const selectedPages = Array.from(checkboxes).map(cb => cb.value);
    
    if (selectedPages.length === 0) return;

    Swal.fire({
        title: 'Confirmar Exclusão',
        html: `
            <div class="text-center">
                <i class="fas fa-exclamation-triangle text-warning display-4 mb-3"></i>
                <p>Você está prestes a excluir <b>${selectedPages.length}</b> página(s).</p>
                <p class="text-danger">Esta ação não pode ser desfeita!</p>
            </div>
        `,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#dc3545',
        cancelButtonColor: '#6c757d',
        confirmButtonText: '<i class="fas fa-trash me-2"></i>Sim, excluir!',
        cancelButtonText: '<i class="fas fa-times me-2"></i>Cancelar',
        reverseButtons: true
    }).then((result) => {
        if (result.isConfirmed) {
            showLoading();
            const form = document.getElementById('delete-selected-form');
            selectedPages.forEach(filename => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'selected_pages[]';
                input.value = filename;
                form.appendChild(input);
            });
            form.submit();
        }
    });
}

// Validação de Formulários
function validateForm(formId) {
    const form = document.getElementById(formId);
    if (!form) return true;

    let isValid = true;
    const requiredFields = form.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            isValid = false;
            field.classList.add('is-invalid');
            
            // Criar mensagem de erro se não existir
            if (!field.nextElementSibling?.classList.contains('invalid-feedback')) {
                const feedback = document.createElement('div');
                feedback.className = 'invalid-feedback';
                feedback.textContent = 'Este campo é obrigatório';
                field.parentNode.appendChild(feedback);
            }
        } else {
            field.classList.remove('is-invalid');
        }
    });

    return isValid;
}

// Função para validar formato da chave PIX
function validatePixKey(pixKey) {
    // Remove espaços e caracteres especiais
    pixKey = pixKey.trim().replace(/[^\w@.-]/g, '');
    
    // Validar CPF (11 dígitos)
    if (/^\d{11}$/.test(pixKey)) return true;
    
    // Validar CNPJ (14 dígitos)
    if (/^\d{14}$/.test(pixKey)) return true;
    
    // Validar email
    if (/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(pixKey)) return true;
    
    // Validar telefone (+5511999999999)
    if (/^\+55\d{10,11}$/.test(pixKey)) return true;
    
    // Validar chave aleatória (UUID)
    if (/^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i.test(pixKey)) return true;
    
    return false;
}

// Função para gerar o código PIX
async function generatePixCode(pixKey, amount, description) {
    try {
        // Validar dados
        if (!pixKey) {
            throw new Error('Chave PIX é obrigatória');
        }

        if (!amount || amount <= 0) {
            throw new Error('Valor deve ser maior que zero');
        }

        // Configurar dados do merchant
        const merchantName = "MATEUS SHOP".replace(/[^A-Za-z0-9 ]/g, '').toUpperCase();
        const merchantCity = "SAO PAULO".replace(/[^A-Za-z0-9 ]/g, '').toUpperCase();
        
        // Gerar código PIX
        const pixData = CopyAndPaste({
            name: merchantName.substring(0, 25),
            key: pixKey.trim(),
            amount: parseFloat(amount),
            city: merchantCity.substring(0, 15),
            id: description ? description.substring(0, 25) : 'PAGAMENTO'
        });

        // Gerar QR Code
        const qrCodeData = QrCodeImage({
            name: merchantName.substring(0, 25),
            key: pixKey.trim(),
            amount: parseFloat(amount),
            city: merchantCity.substring(0, 15),
            id: description ? description.substring(0, 25) : 'PAGAMENTO',
            path: 'qrcodes/'
        });

        return {
            success: true,
            pixCode: pixData.payload,
            qrCodePath: qrCodeData.path
        };

    } catch (error) {
        console.error('Erro ao gerar PIX:', error);
        throw new Error('Erro ao gerar código PIX: ' + error.message);
    }
}

// Função para mostrar o modal de planos
function showPlansModal() {
    const plansContainer = document.getElementById('plansContainer');
    plansContainer.innerHTML = '';
    
    window.planos.forEach((plano, index) => {
        const planCard = document.createElement('div');
        planCard.className = 'col-md-6 mb-3';
        planCard.innerHTML = `
            <div class="plan-card ${window.selectedPlan && window.selectedPlan.valor === plano.valor ? 'selected' : ''}" 
                 onclick="selectPlan(${index})">
                <h5>${plano.nome}</h5>
                <p class="mb-1">R$ ${parseFloat(plano.valor).toFixed(2)}</p>
                <small class="text-muted">${plano.descricao}</small>
            </div>
        `;
        plansContainer.appendChild(planCard);
    });
    
    const plansModal = new bootstrap.Modal(document.getElementById('plansModal'));
    plansModal.show();
}

// Função para selecionar um plano
async function selectPlan(planIndex) {
    try {
        const plano = window.planos[planIndex];
        window.selectedPlan = plano;
        
        const pixKey = document.getElementById('pixKey').getAttribute('data-pix-key');
        const result = await generatePixCode(pixKey, plano.valor, plano.nome);
        
        if (result.success) {
            document.getElementById('pixKey').value = result.pixCode;
            document.getElementById('copyButton').style.display = 'inline-block';
            document.getElementById('qrButton').style.display = 'inline-block';
            
            // Atualizar cards de plano
            document.querySelectorAll('.plan-card').forEach((card, index) => {
                card.classList.toggle('selected', index === planIndex);
            });
            
            // Fechar modal
            const plansModal = bootstrap.Modal.getInstance(document.getElementById('plansModal'));
            plansModal.hide();
        }
    } catch (error) {
        console.error('Erro ao gerar código PIX:', error);
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message
        });
    }
}

// Função para copiar o código PIX
function copiarChavePIX() {
    const pixKey = document.getElementById('pixKey');
    if (pixKey) {
        navigator.clipboard.writeText(pixKey.value)
            .then(() => {
                Swal.fire({
                    icon: 'success',
                    title: 'Copiado!',
                    text: 'Código PIX copiado com sucesso!',
                    timer: 2000,
                    showConfirmButton: false
                });
            })
            .catch(err => {
                console.error('Erro ao copiar:', err);
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: 'Não foi possível copiar o código PIX'
                });
            });
    }
}

// Função para mostrar o QR Code
function showQRCode() {
    const pixKey = document.getElementById('pixKey').getAttribute('data-pix-key');
    const selectedPlan = window.selectedPlan;
    
    if (!selectedPlan) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: 'Por favor, selecione um plano primeiro'
        });
        return;
    }

    try {
        generatePixCode(pixKey, selectedPlan.valor, selectedPlan.nome)
            .then(result => {
                if (result.success) {
                    const qrCodeModal = new bootstrap.Modal(document.getElementById('qrModal'));
                    document.getElementById('qrCodeImage').src = result.qrCodePath;
                    qrCodeModal.show();
                }
            })
            .catch(error => {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: error.message
                });
            });
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message
        });
    }
}

// Event Listeners
document.addEventListener('DOMContentLoaded', () => {
    // Inicializar tooltips
    const tooltips = document.querySelectorAll('[data-bs-toggle="tooltip"]');
    tooltips.forEach(tooltip => new bootstrap.Tooltip(tooltip));
    
    // Validação de formulários
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', (e) => {
            if (!validateForm(form.id)) {
                e.preventDefault();
                showMessage('Erro', 'Por favor, preencha todos os campos obrigatórios', 'error');
            } else {
                showLoading();
            }
        });
    });
    
    // Limpar validação ao digitar
    const inputs = document.querySelectorAll('.form-control, .form-select');
    inputs.forEach(input => {
        input.addEventListener('input', () => {
            input.classList.remove('is-invalid');
        });
    });

    // Aplicar máscara de moeda
    if (typeof $ !== 'undefined') {
        $('.money').mask('#.##0,00', {
            reverse: true,
            placeholder: '0,00'
        });
    }

    // Configurar formulário de planos
    const planosForm = document.getElementById('planosForm');
    if (planosForm) {
        planosForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const valores = document.querySelectorAll('input[name="plano_valor[]"]');
            valores.forEach(input => {
                let valor = input.value.replace(/\./g, '').replace(',', '.');
                valor = parseFloat(valor) * 100;
                input.value = valor;
            });
            showLoading();
            this.submit();
        });
    }

    // Adicionar evento ao botão de adicionar plano
    const addPlanoBtn = document.getElementById('addPlanoBtn');
    if (addPlanoBtn) {
        addPlanoBtn.addEventListener('click', addPlano);
    }

    // Mostrar mensagem do sistema se existir
    if (typeof systemMessage !== 'undefined') {
        showMessage('Aviso', systemMessage);
    }
});

// Função para adicionar novo plano
function addPlano() {
    const container = document.getElementById('planos-container');
    if (!container) return;

    const planoItem = document.createElement('div');
    planoItem.className = 'row mb-3 plano-item';
    planoItem.innerHTML = `
        <div class="col-md-4">
            <label class="form-label">Nome do Plano</label>
            <input type="text" name="plano_nome[]" class="form-control" required>
        </div>
        <div class="col-md-3">
            <label class="form-label">Valor (R$)</label>
            <input type="text" name="plano_valor[]" class="form-control money" required>
            <small class="text-muted">Digite o valor (ex: 35,00 para R$ 35,00)</small>
        </div>
        <div class="col-md-4">
            <label class="form-label">Descrição</label>
            <input type="text" name="plano_descricao[]" class="form-control" required>
        </div>
        <div class="col-md-1 d-flex align-items-end">
            <button type="button" class="btn btn-danger btn-sm" onclick="removePlano(this)">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;

    container.appendChild(planoItem);

    // Aplicar máscara de moeda ao novo campo
    if (typeof $ !== 'undefined') {
        $(planoItem).find('.money').mask('#.##0,00', {
            reverse: true,
            placeholder: '0,00'
        });
    }
}

// Função para remover plano
function removePlano(button) {
    const planoItem = button.closest('.plano-item');
    if (planoItem) {
        // Verificar se é o último plano
        const totalPlanos = document.querySelectorAll('.plano-item').length;
        if (totalPlanos > 1) {
            planoItem.remove();
        } else {
            showMessage('Aviso', 'Você deve manter pelo menos um plano', 'warning');
        }
    }
} 